<?php

namespace App\Http\Controllers;

use App\Models\TrainingRequest;
use Illuminate\Http\Request;

class TrainingRequestController extends Controller
{
    // batas biaya untuk butuh Direktur
    private const DIRECTOR_THRESHOLD = 20000000;

    // LIST
    public function index(Request $request)
    {
        $user = $request->user();

        if ($user->isHrd() || $user->isAdmin()) {
            $query = TrainingRequest::with('user');
        } else {
            $query = TrainingRequest::with('user')
                ->where('user_id', $user->id);
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        return $query->orderByDesc('created_at')->get();
    }

    // CREATE (user)
    public function store(Request $request)
    {
        $user = $request->user();

        $data = $request->validate([
            'title'           => ['required', 'string', 'max:255'],
            'description'     => ['nullable', 'string'],
            'training_date'   => ['required', 'date'],
            'vendor'          => ['nullable', 'string', 'max:255'],
            'cost'            => ['required', 'integer', 'min:0'],
            'purposes'        => ['nullable', 'array'],
            'purposes.*'      => ['string'],
            'expected_result' => ['nullable', 'string'],
        ]);

        $data['user_id'] = $user->id;
        $data['status']  = 'submitted';

        $training = TrainingRequest::create($data);

        return response()->json($training->load('user'), 201);
    }

    // DETAIL
    public function show(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if ($user->isUser() && $trainingRequest->user_id !== $user->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        return $trainingRequest->load('user');
    }

    // UPDATE (oleh pemilik, hanya submitted)
    public function update(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if ($trainingRequest->user_id !== $user->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'submitted') {
            return response()->json([
                'message' => 'Tidak bisa edit request yang sudah di-approve/reject',
            ], 422);
        }

        $data = $request->validate([
            'title'           => ['required', 'string', 'max:255'],
            'description'     => ['nullable', 'string'],
            'training_date'   => ['required', 'date'],
            'vendor'          => ['nullable', 'string', 'max:255'],
            'cost'            => ['required', 'integer', 'min:0'],
            'purposes'        => ['nullable', 'array'],
            'purposes.*'      => ['string'],
            'expected_result' => ['nullable', 'string'],
        ]);

        $trainingRequest->update($data);

        return $trainingRequest->refresh()->load('user');
    }

    // DELETE (oleh pemilik, hanya submitted)
    public function destroy(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if ($trainingRequest->user_id !== $user->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'submitted') {
            return response()->json([
                'message' => 'Tidak bisa hapus request yang sudah di-approve/reject',
            ], 422);
        }

        $trainingRequest->delete();

        return response()->json(['message' => 'Deleted']);
    }

    // APPROVE HRD
    public function approveHrd(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if (! $user->isHrd()) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'submitted') {
            return response()->json([
                'message' => 'Hanya request dengan status submitted yang bisa di-approve HRD',
            ], 422);
        }

        $data = $request->validate([
            'hrd_notes' => ['nullable', 'string'],
        ]);

        $trainingRequest->update([
            'status'             => 'approved_hrd',
            'hrd_notes'          => $data['hrd_notes'] ?? $trainingRequest->hrd_notes,
            'approved_by_hrd'    => $user->id,
            'approved_at_hrd'    => now(),
        ]);

        return $trainingRequest->refresh()->load('user');
    }

    // REJECT HRD
    public function rejectHrd(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if (! $user->isHrd()) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'submitted') {
            return response()->json([
                'message' => 'Hanya request dengan status submitted yang bisa di-reject HRD',
            ], 422);
        }

        $data = $request->validate([
            'hrd_notes' => ['nullable', 'string'],
        ]);

        $trainingRequest->update([
            'status'             => 'rejected_hrd',
            'hrd_notes'          => $data['hrd_notes'] ?? $trainingRequest->hrd_notes,
            'approved_by_hrd'    => $user->id,
            'approved_at_hrd'    => now(),
        ]);

        return $trainingRequest->refresh()->load('user');
    }

    // APPROVE DIREKTUR (role: admin), hanya jika cost > 20jt & sudah approved_hrd
    public function approveDirector(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if (! $user->isAdmin()) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'approved_hrd') {
            return response()->json([
                'message' => 'Direktur hanya bisa approve request yang sudah disetujui HRD',
            ], 422);
        }

        if ($trainingRequest->cost <= self::DIRECTOR_THRESHOLD) {
            return response()->json([
                'message' => 'Request dengan biaya ≤ 20.000.000 tidak memerlukan persetujuan Direktur',
            ], 422);
        }

        $trainingRequest->update([
            'status'                 => 'approved_director',
            'approved_by_director'   => $user->id,
            'approved_at_director'   => now(),
        ]);

        return $trainingRequest->refresh()->load('user');
    }

    // REJECT DIREKTUR
    public function rejectDirector(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if (! $user->isAdmin()) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'approved_hrd') {
            return response()->json([
                'message' => 'Direktur hanya bisa reject request yang sudah disetujui HRD',
            ], 422);
        }

        if ($trainingRequest->cost <= self::DIRECTOR_THRESHOLD) {
            return response()->json([
                'message' => 'Request dengan biaya ≤ 20.000.000 tidak memerlukan persetujuan Direktur',
            ], 422);
        }

        $trainingRequest->update([
            'status'                 => 'rejected_director',
            'approved_by_director'   => $user->id,
            'approved_at_director'   => now(),
        ]);

        return $trainingRequest->refresh()->load('user');
    }
}